#This code estimates the ash volume spatial distribution using kriging with esternal drift. 
#As dependent variable, we the observed #ash volume data from the national forest inventory  (donnees brutes). 
#The data are subdivided into five subsamples according to the forest cover type in which the ash are observed.
#The forest type of the sample plot are: 1 broadleaves high forest, 2 broadleaves high forest & coppice, 3 broadleaves coppice, 
#   5 conifers, 9 other.
# For each forest type, we fit a linear regression model using the following independent variables (see paper for full references):
# 1)the soil water deficit as defined in Piedallu et al (2016), obtained from the portal on forest spatial data SILVAE (LERFoB and IFN, 2008)
# 2)the soil pH obtained from the portal on forest spatial data SILVAE (LERFoB and IFN, 2008), 
# 3)the elevation (Institut national de l'information gographique et forestire - IGN)
# 4)a categorical variable representing the 86 diferent types of ecological regions present in French forests (SER), obrained from
#   Institut national de l'information gographique et forestire - IGN
#
# All these data were aggregated to a single spatial dataset for each forest type (allplot_ft1, allplot_ft2, allplot_ft3, allplot_ft5, 
# allplot_ft9), Once the regression models are estimated, the we plot the experimental variogram and fit a theorical variogram to the data
# 
# Using the estimated models, we predict the forest volume distribution for a 1000x1000 m grid covering the entire french territory.
# Using the independent variables spatial data, we assign to each pixel a value for the independent variable (covars.tif). All the pixel
# not containing forest areas were set as NA
# 
# Once we obrained the raster with the agerage volume per hectare for each pixel (within each forest type), we multiply it by the 
# raster with the total forest area per pixel (within each forest type)

rm(list = ls())


# spatial
library(raster)
library(rgdal)
library(sp)

#geostatistics
library(gstat)

#input repository
input_dir<-c("~/LEF/Dottorato/Case_studies/Chalara/Data/Ash_distribution/FFSM_input/Kriging/Plots/")
covar_dir<-c("~/LEF/Dottorato/Case_studies/Chalara/Data/Ash_distribution/FFSM_input/Kriging/Covariates/")
area_dir<-c("~/LEF/Dottorato/Case_studies/Chalara/Data/Ash_distribution/FFSM_input/Forest_area/Raster/")

#output repository
output_dir<-c("~/LEF/Dottorato/Case_studies/Chalara/Data/Ash_distribution/FFSM_input/Kriging/Results/Output_ked/")

#========================================================================================================================
# KRIGINg WITH SER
#========================================================================================================================

#====== FTYPE 1 - Broadleaved high forest ===============================================================================

#import sample plot data data

Ash_START <- read.csv(paste0(input_dir,"allplot_ft1.csv"))
dat<-data.frame(x=Ash_START$X, y=Ash_START$Y, vol=Ash_START$vol, ftype=factor(Ash_START$ftype_plot), elev=Ash_START$elev, 
                soil=factor(Ash_START$soil), sret=Ash_START$sret3035, yret=Ash_START$yret3035, ser=factor(Ash_START$numser),
                wstress=Ash_START$wstress, ph=Ash_START$ph)
coordinates(dat)<-c('x','y')
projection(dat)<- CRS("+init=epsg:3035")

summary(lm(vol~ser + elev + ph + wstress, data=dat))

v_cov<-variogram(vol~ser + elev + ph + wstress  ,data=dat,  cutoff=30000)
print(plot(v_cov, pl = T))

vico <- fit.variogram(v_cov, vgm(1500, "Exp", 800,0))



#import grid and covariates

grid<- readGDAL(paste0(covar_dir,"covars.tif"))
grid$band1[which(grid$band1==0)] = NA
grid$band2[which(grid$band1==0)] = NA
grid$band3[which(grid$band1==0)] = NA
grid$band4[which(grid$band1==0)] = NA
grid$band5[which(grid$band1==0)] = NA
names(grid)<-c("area","ser", "elev", "wstress", "ph")
grid$ser=as.factor(grid$ser)
projection(grid)<- CRS("+init=epsg:3035")



ked <- krige(vol ~  ser + elev + ph + wstress, loc=dat,  newdata=grid, model=vico)

#ramp<-seq(from=0, to=150, by=5)


print(spplot(ked, "var1.pred", asp=1,  col.regions=bpy.colors(64), main="Kriging with external drift"))
writeGDAL(ked, paste0(output_dir,"kirg_ft1.tif"))






#====== FTYPE 2 - Broadleaved mixed high forest and coppice ===============================================================


#import sample plot data data
Ash_START <- read.csv(paste0(input_dir,"allplot_ft2.csv"))
dat<-data.frame(x=Ash_START$X, y=Ash_START$Y, vol=Ash_START$vol, ftype=factor(Ash_START$ftype_plot), elev=Ash_START$elev, 
                soil=factor(Ash_START$soil), sret=Ash_START$sret3035, yret=Ash_START$yret3035, ser=factor(Ash_START$numser),
                wstress=Ash_START$wstress, ph=Ash_START$ph)
coordinates(dat)<-c('x','y')
projection(dat)<- CRS("+init=epsg:3035")

summary(lm(vol~ ser + elev + ph + wstress , data=dat))
summary(lm(vol~ elev + ph + wstress , data=dat))


v_cov<-variogram(vol~ser + elev + ph + wstress ,data=dat,  cutoff=25000)
print(plot(v_cov, pl = T))

vico <- fit.variogram(v_cov, vgm(800, "Exp", 1300,0))


ked <- krige(vol ~  ser + elev + ph + wstress, loc=dat,  newdata=grid, model=vico)

ramp<-seq(from=0, to=150, by=5)

writeGDAL(ked, paste0(output_dir,"kirg_ft2.tif"))







#====== FTYPE 3  - Broadleaved coppice   ==================================================================================


#import sample plot data data
Ash_START <- read.csv(paste0(input_dir,"allplot_ft3.csv"))

dat<-data.frame(x=Ash_START$X, y=Ash_START$Y, vol=Ash_START$vol, ftype=factor(Ash_START$ftype_plot), elev=Ash_START$elev, 
                soil=factor(Ash_START$soil), sret=Ash_START$sret3035, yret=Ash_START$yret3035, ser=factor(Ash_START$numser),
                wstress=Ash_START$wstress, ph=Ash_START$ph)
coordinates(dat)<-c('x','y')
projection(dat)<- CRS("+init=epsg:3035")

summary(lm(vol~ ser + elev + ph + wstress , data=dat))

summary(lm(vol~ elev + ph + wstress , data=dat))


v_cov<-variogram(vol~ ser + elev + ph + wstress ,data=dat,  cutoff=150000)
print(plot(v_cov, pl = T))

vico <- fit.variogram(v_cov, vgm(300, "Sph", 20000,0))
print(plot(v_cov, pl = T, model = vico))




#using proxy for ser not observed
grid$ser[which(grid$ser==8)] =7
grid$ser[which(grid$ser==28)] =30
grid$ser[which(grid$ser==29)] =18
grid$ser[which(grid$ser==33)] =32
grid$ser[which(grid$ser==71)] =69


# 8, 28, 29, 33, 71

ked <- krige(vol ~  ser + elev + ph + wstress, loc=dat,  newdata=grid, model=vico)

ramp<-seq(from=0, to=150, by=5)


writeGDAL(ked, paste0(output_dir,"kirg_ft3.tif"))






#====== FTYPE 5 - Conifers ================================================================================================

#import sample plot data data

Ash_START <- read.csv(paste0(input_dir,"allplot_ft5.csv"))
dat<-data.frame(x=Ash_START$X, y=Ash_START$Y, vol=Ash_START$vol, ftype=factor(Ash_START$ftype_plot), elev=Ash_START$elev, 
                soil=factor(Ash_START$soil), sret=Ash_START$sret3035, yret=Ash_START$yret3035, ser=factor(Ash_START$numser),
                wstress=Ash_START$wstress, ph=Ash_START$ph)
coordinates(dat)<-c('x','y')
projection(dat)<- CRS("+init=epsg:3035")

summary(lm(vol~ ser + elev + ph + wstress  , data=dat))
summary(lm(vol~  elev + ph + wstress  , data=dat))


v_cov<-variogram(vol~ ser + elev + ph + wstress  ,data=dat,  cutoff=30000)
print(plot(v_cov, pl = T))

vico <- fit.variogram(v_cov, vgm(120, "Sph", 5000,0))
print(plot(v_cov, pl = T, model = vico))


#import grid and covariates

grid<- readGDAL(paste0(covar_dir,"covars.tif"))
grid$band1[which(grid$band1==0)] = NA
grid$band2[which(grid$band1==0)] = NA
grid$band3[which(grid$band1==0)] = NA
grid$band4[which(grid$band1==0)] = NA
grid$band5[which(grid$band1==0)] = NA
names(grid)<-c("area","ser", "elev", "wstress", "ph")
grid$ser=as.factor(grid$ser)
projection(grid)<- CRS("+init=epsg:3035")



ked <- krige(vol ~  ser + elev + ph + wstress, loc=dat,  newdata=grid, model=vico)

ramp<-seq(from=0, to=150, by=5)



writeGDAL(ked, paste0(output_dir,"kirg_ft5.tif"))




#====== FTYPE 9 - Other =====================================================================================================

#import sample plot data data

Ash_START <- read.csv(paste0(input_dir,"allplot_ft9.csv"))
dat<-data.frame(x=Ash_START$X, y=Ash_START$Y, vol=Ash_START$vol, ftype=factor(Ash_START$ftype_plot), elev=Ash_START$elev, 
                soil=factor(Ash_START$soil), sret=Ash_START$sret3035, yret=Ash_START$yret3035, ser=factor(Ash_START$numser),
                wstress=Ash_START$wstress, ph=Ash_START$ph)
coordinates(dat)<-c('x','y')
projection(dat)<- CRS("+init=epsg:3035")

summary(lm(vol~ ser + elev + ph + wstress , data=dat))

summary(lm(vol~ elev + ph + wstress , data=dat))


v_cov<-variogram(vol~ ser + elev + ph + wstress  ,data=dat,  cutoff=300000)
print(plot(v_cov, pl = T))

vico <- fit.variogram(v_cov, vgm(500, "Gau", 150000,0))
print(plot(v_cov, pl = T, model = vico))



#using proxy for ser not observed
grid$ser[which(grid$ser==10)] =9
grid$ser[which(grid$ser==11)] =1
grid$ser[which(grid$ser==29)] =18
grid$ser[which(grid$ser==33)] =32
grid$ser[which(grid$ser==42)] =43
grid$ser[which(grid$ser==46)] =45
grid$ser[which(grid$ser==51)] =50
grid$ser[which(grid$ser==53)] =52
grid$ser[which(grid$ser==57)] =30



ked <- krige(vol ~  ser + elev + ph + wstress, loc=dat,  newdata=grid, model=vico)

ramp<-seq(from=0, to=150, by=5)

#10, 11, 29, 33, 42, 46, 51, 53, 57


writeGDAL(ked, paste0(output_dir,"kirg_ft9.tif"))




#========================================================================================================================
# TOTAL VOLUME PER PIXEL
#========================================================================================================================



######### FTYPE1

ked<-raster(paste0(output_dir,"kirg_ft1.tif"), band=1)

v_ha=ked
v_ha[v_ha<0]<-0

area=raster(paste0(area_dir,"area1"))
projection(area)<- CRS("+init=epsg:3035")

vol=v_ha*area

writeRaster(vol, filename=paste0(output_dir,"vol1.tif"), format="GTiff", overwrite=TRUE)
cellStats(vol, 'sum')

################## FTYPE2

ked<-raster(paste0(output_dir,"kirg_ft2.tif"), band=1)

v_ha=ked
v_ha[v_ha<0]<-0

area=raster(paste0(area_dir,"area2"))
projection(area)<- CRS("+init=epsg:3035")

vol=v_ha*area

writeRaster(vol, filename=paste0(output_dir,"vol2.tif"), format="GTiff", overwrite=TRUE)
cellStats(vol, 'sum')

##################  FTYPE3
ked<-raster(paste0(output_dir,"kirg_ft3.tif"), band=1)

v_ha=ked
v_ha[v_ha<0]<-0

area=raster(paste0(area_dir,"area3"))
projection(area)<- CRS("+init=epsg:3035")

vol=v_ha*area

writeRaster(vol, filename=paste0(output_dir,"vol3.tif"), format="GTiff", overwrite=TRUE)
cellStats(vol, 'sum')



##################  FTYPE5
ked<-raster(paste0(output_dir,"kirg_ft5.tif"), band=1)

v_ha=ked
v_ha[v_ha<0]<-0

area=raster(paste0(area_dir,"area5"))
projection(area)<- CRS("+init=epsg:3035")

vol=v_ha*area

writeRaster(vol, filename=paste0(output_dir,"vol5.tif"), format="GTiff", overwrite=TRUE)
cellStats(vol, 'sum')




################## FTYPE9
ked<-raster(paste0(output_dir,"kirg_ft9.tif"), band=1)

v_ha=ked
v_ha[v_ha<0]<-0

area=raster(paste0(area_dir,"area9"))
projection(area)<- CRS("+init=epsg:3035")

vol=v_ha*area

writeRaster(vol, filename=paste0(output_dir,"vol9.tif"), format="GTiff", overwrite=TRUE)
cellStats(vol, 'sum')









